package com.printer.sdk.demo.util;

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.ColorMatrix;
import android.graphics.ColorMatrixColorFilter;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Typeface;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * @author mxlei
 * @date 2020/9/16
 */
public class BitmapUtil {

    /**
     * 转换Bitmap为黑白图片(Floyd-Steinberg)，图片颜色只有0x000000和0xffffff两种
     *
     * @param img   需要转换的图片
     * @param shake 是否抖动模式
     */
    public static Bitmap thresholdBitmap(Bitmap img, boolean shake) {
        //转灰色图
        int height = img.getHeight();
        int width = img.getWidth();
        int[] pixels = thresholdBitmapPixels(img, shake);
        Bitmap mBitmap = Bitmap.createBitmap(width, height, img.getConfig());
        mBitmap.setPixels(pixels, 0, width, 0, 0, width, height);
        return mBitmap;
    }

    /**
     * 转换图片为灰色图，效果比是要Android的ColorFilter差，但是执行效率高
     */
    public static int[] convertGrayBitmap(Bitmap img) {
        int width = img.getWidth();         //获取位图的宽
        int height = img.getHeight();       //获取位图的高
        int[] pixels = new int[width * height]; //通过位图的大小创建像素点数组
        img.getPixels(pixels, 0, width, 0, 0, width, height);
        int alpha = 0xFF << 24;
        for (int i = 0; i < height; i++) {
            for (int j = 0; j < width; j++) {
                int grey = pixels[width * i + j];

                int red = ((grey & 0x00FF0000) >> 16);
                int green = ((grey & 0x0000FF00) >> 8);
                int blue = (grey & 0x000000FF);

                grey = (int) (red * 0.3f + green * 0.59f + blue * 0.11f);
                grey = alpha | (grey << 16) | (grey << 8) | grey;
                pixels[width * i + j] = grey;
            }
        }
        return pixels;
    }

    public static Bitmap convertGrayBitmapByColorFilter(Bitmap bitmap) {
        //转灰色图
        int height = bitmap.getHeight();
        int width = bitmap.getWidth();
        Bitmap grayBitmap = Bitmap.createBitmap(width, height, Bitmap.Config.RGB_565);
        Canvas c = new Canvas(grayBitmap);
        c.drawColor(Color.WHITE);
        Paint paint = new Paint();
        ColorMatrix cm = new ColorMatrix();
        cm.setSaturation(0.0F);
        ColorMatrixColorFilter f = new ColorMatrixColorFilter(cm);
        paint.setColorFilter(f);
        c.drawBitmap(bitmap, 0.0F, 0.0F, paint);
        return grayBitmap;
    }

    public static int[] thresholdBitmapPixels(Bitmap bitmap, boolean shake) {
        bitmap = convertGrayBitmapByColorFilter(bitmap);
        int width = bitmap.getWidth();
        int height = bitmap.getHeight();
        //通过位图的大小创建像素点数组
        int[] pixels = new int[width * height];
        bitmap.getPixels(pixels, 0, width, 0, 0, width, height);
//        int[] pixels = convertGrayBitmap(bitmap);
        if (shake) {
            int[] grays = Arrays.copyOf(pixels, pixels.length);
            for (int i = 0; i < grays.length; i++) {
                grays[i] = (grays[i] >> 16) & 0xFF;
            }
            int e;
            for (int i = 0; i < height; i++) {
                for (int j = 0; j < width; j++) {
                    int px = grays[width * i + j] & 0xFF;
                    if (px >= 128) {
                        pixels[width * i + j] = 0xffffffff;
                        e = px - 255;
                    } else {
                        pixels[width * i + j] = 0xff000000;
                        e = px - 0;
                    }
                    //处理抖动
                    if (j < width - 1 && i < height - 1) {
                        //右边像素处理
                        grays[width * i + j + 1] += 7 * e / 16;
                        //下
                        grays[width * (i + 1) + j] += 5 * e / 16;
                        //右下
                        grays[width * (i + 1) + j + 1] += e / 16;
                        //左下
                        if (j > 0) {
                            grays[width * (i + 1) + j - 1] += 3 * e / 16;
                        }
                    } else if (j == width - 1 && i < height - 1) {
                        //下方像素处理
                        grays[width * (i + 1) + j] += 5 * e / 16;
                    } else if (j < width - 1 && i == height - 1) {
                        //右边像素处理
                        grays[width * (i) + j + 1] += 7 * e / 16;
                    }
                }
            }
        } else {
            int r, g, b, gr;
            for (int i = 0; i < pixels.length; i++) {
                r = (pixels[i] >> 16) & 0xFF;
                g = (pixels[i] >> 8) & 0xFF;
                b = pixels[i] & 0xFF;
                gr = (int) (r * 0.3f + g * 0.59f + b * 0.11f);
                if (gr >= 128) {
                    pixels[i] = 0xffffffff;
                } else {
                    pixels[i] = 0xff000000;
                }
            }
        }
        return pixels;
    }

    /**
     * 把图片转换为可打印的数据格式
     *
     * @param bitmap 图片对象
     */
    public static byte[] getDotMatrixOfBitmapForPrint(Bitmap bitmap, boolean shake) {
        return getDotMatrixOfBitmapForPrint(bitmap, shake, true);
    }

    public static byte[] getDotMatrixOfBitmapForPrint(Bitmap bitmap, boolean shake, boolean reverseBit) {
        int columnSize = (int) Math.ceil(bitmap.getWidth() / 8.0);
        byte[] data = new byte[columnSize * bitmap.getHeight()];
        int dataIndex = 0;
        int dataBit, lineIndex;
        int A, R, G, B, x, y, GR;
        int pixelColor;
        int height = bitmap.getHeight();
        int width = bitmap.getWidth();

        int[] pixels = thresholdBitmapPixels(bitmap, shake);
        for (y = 0; y < height; y++) {
            for (x = 0; x < columnSize; x++) {
                for (dataBit = 0; dataBit < 8; dataBit++) {
                    lineIndex = x * 8 + dataBit;
                    if (lineIndex < width) {
                        pixelColor = pixels[y * width + lineIndex];
                        R = (pixelColor >> 16) & 0xFF;
                        G = (pixelColor >> 8) & 0xFF;
                        B = pixelColor & 0xFF;
                        GR = (int) (R * 0.3f + G * 0.59f + B * 0.11f);
                        if (GR < 128) {
                            if (reverseBit) {
                                data[dataIndex] = (byte) ((data[dataIndex] << 1) | 0x01);
                            } else {
                                data[dataIndex] = (byte) ((data[dataIndex] << 1) & 0xFE);
                            }
                        } else {
                            if (reverseBit) {
                                data[dataIndex] = (byte) ((data[dataIndex] << 1) & 0xFE);
                            } else {
                                data[dataIndex] = (byte) ((data[dataIndex] << 1) | 0x01);
                            }
                        }
                    }
                }
                dataIndex++;
            }
        }
        return data;
    }

    public static byte[] getDotMatrixOfBitmapForPrintVertical(Bitmap bitmap, boolean shake) {
        byte[] data = new byte[bitmap.getHeight() * bitmap.getWidth() / 8];
        int dataIndex = 0;
        int dataBit = 0;
        int height = bitmap.getHeight();
        int width = bitmap.getWidth();

        int[] pixels = thresholdBitmapPixels(bitmap, shake);
        for (int x = 0; x < width; x++) {
            for (int y = 0; y < height; ) {
                if ((pixels[width * y + x] & 0xFF) > 127) {
                    dataBit = ((dataBit << 1) & 0xFE);
                } else {
                    dataBit = ((dataBit << 1) | 0x01);
                }
                y++;
                if (y % 8 == 0) {
                    data[dataIndex++] = (byte) dataBit;
                    dataBit = 0;
                }
            }
        }
        return data;
    }

    public static List<Bitmap> cutBitmapByHeight(Bitmap bitmap, int maxHeight) {
        List<Bitmap> list = new ArrayList<>();
        if (bitmap == null || maxHeight < 1) {
            return list;
        }
        for (int startHeight = 0; startHeight < bitmap.getHeight(); startHeight += maxHeight) {
            int endHeight = startHeight + maxHeight;
            if (endHeight > bitmap.getHeight()) {
                endHeight = bitmap.getHeight();
            }
            Bitmap sub = Bitmap.createBitmap(bitmap, 0, startHeight, bitmap.getWidth(), endHeight - startHeight);
            list.add(sub);
        }
        return list;
    }

    /**
     * 缩放图片
     *
     * @param bitmap    源图片
     * @param dstWidth  目标图宽度
     * @param dstHeight 目标图高度
     * @return 缩放后的图片
     */
    public static Bitmap resizeBitmap(Bitmap bitmap, int dstWidth, int dstHeight) {
        return resizeBitmap(bitmap, dstWidth, dstHeight, true);
    }

    /**
     * 缩放图片
     *
     * @param bitmap        源图片
     * @param dstWidth      目标图宽度
     * @param dstHeight     目标图高度
     * @param stretchBitmap 是否拉伸图片
     * @return 缩放后的图片
     */
    public static Bitmap resizeBitmap(Bitmap bitmap, int dstWidth, int dstHeight, boolean stretchBitmap) {
        if (bitmap == null) {
            return null;
        }
        if (bitmap.getWidth() == dstWidth && bitmap.getHeight() == dstHeight) {
            return bitmap;
        }
        if (dstWidth < 1 || dstHeight < 1) {
            throw new IllegalArgumentException("Bitmap output width and height must greater than 1");
        }
        if (stretchBitmap) {
            float scaleX = ((float) dstWidth) / bitmap.getWidth();
            float scaleY = ((float) dstHeight) / bitmap.getHeight();
            Matrix matrix = new Matrix();
            matrix.postScale(scaleX, scaleY);
            bitmap = Bitmap.createBitmap(bitmap, 0, 0, bitmap.getWidth(), bitmap.getHeight(), matrix, true);
        } else {
            Bitmap paddingBitmap = Bitmap.createBitmap(dstWidth, dstHeight, bitmap.getConfig());
            Canvas canvas = new Canvas(paddingBitmap);
            canvas.drawColor(Color.WHITE);
            canvas.drawBitmap(bitmap, 0, 0, null);
            bitmap = paddingBitmap;
        }
        return bitmap;
    }

    /**
     * 旋转图片
     *
     * @param bitmap   图片
     * @param rotation 旋转角度
     */
    public static Bitmap rotateBitmap(Bitmap bitmap, float rotation) {
        if (bitmap == null) {
            return null;
        }
        if (rotation % 360 == 0) {
            return bitmap;
        }
        Matrix matrix = new Matrix();
        matrix.postRotate(rotation);
        return Bitmap.createBitmap(bitmap, 0, 0, bitmap.getWidth(), bitmap.getHeight(), matrix, true);
    }

    /**
     * 合并多张图片
     *
     * @param verticalMerge 是否垂直合并
     * @param bitmaps       图片列表
     * @return 合并后的图片
     */
    public static Bitmap mergeBitmap(boolean verticalMerge, Bitmap... bitmaps) {
        if (bitmaps == null || bitmaps.length == 0) {
            return null;
        }
        if (bitmaps.length == 1) {
            return bitmaps[0];
        }
        if (verticalMerge) {
            int height = 0;
            int width = 0;
            int y = 0;
            for (Bitmap b : bitmaps) {
                width = Math.max(width, b.getWidth());
                height = height + b.getHeight();
            }
            Bitmap bitmap = Bitmap.createBitmap(width, height, bitmaps[0].getConfig());
            Canvas canvas = new Canvas(bitmap);
            for (Bitmap b : bitmaps) {
                canvas.drawBitmap(b, 0, y, null);
                y += b.getHeight();
            }
            return bitmap;
        } else {
            int height = 0;
            int width = 0;
            int x = 0;
            for (Bitmap b : bitmaps) {
                width = width + b.getWidth();
                height = Math.max(height, b.getHeight());
            }
            Bitmap bitmap = Bitmap.createBitmap(width, height, bitmaps[0].getConfig());
            Canvas canvas = new Canvas(bitmap);
            for (Bitmap b : bitmaps) {
                canvas.drawBitmap(b, x, 0, null);
                x += b.getWidth();
            }
            return bitmap;
        }
    }


    /**
     * 以图片的方式打印文字，可以设置为任意字体、任意字号
     */
    public static Bitmap drawTextToBitmap(int color, String text, int fontSize, boolean bold, boolean underline, Typeface typeface) {
        Paint paint = new Paint();
        typeface = typeface == null ? Typeface.DEFAULT : typeface;
        paint.setTextSize(fontSize);
        if (bold) {
            typeface = Typeface.create(typeface, Typeface.BOLD);
        }

        paint.setTypeface(typeface);

        Bitmap bitmap = Bitmap.createBitmap((int) Math.ceil(paint.measureText(text)), (int) Math.ceil(paint.getFontMetrics().bottom - paint.getFontMetrics().top), Bitmap.Config.RGB_565);
        Canvas canvas = new Canvas(bitmap);
        canvas.drawColor(Color.WHITE);
        paint.setColor(color);
        canvas.drawText(text, 0, -paint.getFontMetrics().top, paint);
        if (underline) {
            int lineY = (int) (bitmap.getHeight() - 2);
            paint.setStrokeWidth(1);
            canvas.drawLine(0, lineY, bitmap.getWidth(), lineY, paint);
        }
        return bitmap;
    }

    /**
     * 绘制文字水印
     */
    public static Bitmap waterMaskText(String text, int fontSize, boolean bold, boolean underline, int blackness, Typeface typeface) {
        if (blackness < 0) {
            blackness = 0;
        }
        if (blackness > 255) {
            blackness = 255;
        }
        Bitmap bitmap = drawTextToBitmap(Color.argb(blackness, 0x7f, 0x7f, 0x7f), text, fontSize, bold, underline, typeface);
        return thresholdBitmap(bitmap, true);
    }

    /**
     * 转换图片水印效果
     */
    public static Bitmap waterMaskBitmap(Bitmap bitmap, int blackness) {
        if (blackness < 0) {
            blackness = 0;
        }
        if (blackness > 255) {
            blackness = 255;
        }
        bitmap = convertGrayBitmapByColorFilter(bitmap);
        Paint paint = new Paint();
        paint.setAlpha(blackness);
        Bitmap dst = Bitmap.createBitmap(bitmap.getWidth(), bitmap.getHeight(), bitmap.getConfig());
        Canvas canvas = new Canvas(dst);
        canvas.drawBitmap(bitmap, 0, 0, paint);
        return thresholdBitmap(dst, true);
    }

}
