package com.printer.sdk.demo.ui;

import android.Manifest;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;
import android.view.View;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.printer.sdk.demo.adapter.PrinterConnectAdapter;
import com.printer.sdk.demo.databinding.ActivityPrinterConnectBinding;
import com.printer.sdk.demo.util.OnRecyclerViewItemClickListener;
import com.printer.sdk2.ConnectionListener;
import com.printer.sdk2.Printer;
import com.printer.sdk2.PrinterFinder;
import com.printer.sdk2.model.BluetoothPrinterDevice;
import com.printer.sdk2.model.PrinterDevice;
import com.printer.sdk2.model.SerialPortPrinterDevice;
import com.printer.sdk2.model.UsbAccessoryPrinterDevice;
import com.printer.sdk2.model.UsbPrinterDevice;
import com.printer.sdk2.model.WifiPrinterDevice;

public class PrinterConnectActivity extends AppCompatActivity implements PrinterFinder.SearchPrinterResultListener, ConnectionListener {

    private final PrinterFinder finder = new PrinterFinder();
    private ActivityPrinterConnectBinding binding;
    private PrinterConnectAdapter adapter;
    private LoadingDialog loadingDialog;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        adapter = new PrinterConnectAdapter();
        loadingDialog = new LoadingDialog(this);
        binding = ActivityPrinterConnectBinding.inflate(getLayoutInflater());
        setContentView(binding.getRoot());
        binding.toolbar.setNavigationOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                finish();
            }
        });
        binding.recyclerview.setLayoutManager(new LinearLayoutManager(this));
        binding.recyclerview.setAdapter(adapter);
        binding.recyclerview.addOnItemTouchListener(new OnRecyclerViewItemClickListener(binding.recyclerview) {
            @Override
            protected void onItemClick(RecyclerView.ViewHolder vh, int position) {
                loadingDialog.show();
                PrinterDevice printerDevice = adapter.getItem(position);
                // connect printer
                Printer.connect(printerDevice);
            }
        });
        // add printer connection listener
        Printer.addConnectionListener(this);
        // find printer devices(bluetooth wifi usb serial_port)

        if (checkBluetoothPermission()) {
            finder.searchPrinters(this);
        }
    }

    private boolean checkBluetoothPermission() {
        String[] pms;
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            return true;
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            pms = new String[]{
                    Manifest.permission.BLUETOOTH_SCAN,
                    Manifest.permission.BLUETOOTH_CONNECT,
                    Manifest.permission.ACCESS_FINE_LOCATION
            };
        } else {
            pms = new String[]{
                    Manifest.permission.BLUETOOTH,
                    Manifest.permission.ACCESS_FINE_LOCATION
            };
        }
        for (String pm : pms) {
            if (checkSelfPermission(pm) != PackageManager.PERMISSION_GRANTED) {
                requestPermissions(pms, 100);
                return false;
            }
        }
        return true;
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (requestCode == 100) {
            boolean grantAll = true;
            for (int r : grantResults) {
                if (r != PackageManager.PERMISSION_GRANTED) {
                    grantAll = false;
                    break;
                }
            }
            if (grantAll) {
                finder.searchPrinters(this);
            }
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        // stop find printer device
        finder.stopSearchDevice();
        // remove printer connection listener
        Printer.removeConnectionListener(this);
    }

    @Override
    public void onSearchBluetoothPrinter(BluetoothPrinterDevice device) {
        adapter.addItem(device);
    }

    @Override
    public void onSearchUsbPrinter(UsbPrinterDevice device) {
        adapter.addItem(device);
    }

    @Override
    public void onSearchUsbPrinter(UsbAccessoryPrinterDevice device) {
        adapter.addItem(device);
    }

    @Override
    public void onSearchNetworkPrinter(WifiPrinterDevice device) {
        adapter.addItem(device);
    }

    @Override
    public void onSearchSerialPortPrinter(SerialPortPrinterDevice device) {
        adapter.addItem(device);
    }

    @Override
    public void onSearchCompleted() {

    }

    @Override
    public void onPrinterConnected(Printer printer) {
        loadingDialog.dismiss();
        finish();
        Toast.makeText(this, "连接成功", Toast.LENGTH_SHORT).show();
    }

    @Override
    public void onPrinterConnectFail(Printer printer) {
        loadingDialog.dismiss();
        Toast.makeText(this, "连接失败", Toast.LENGTH_SHORT).show();
    }

    @Override
    public void onPrinterDisconnect(Printer printer) {
        loadingDialog.dismiss();
        Toast.makeText(this, "连接断开", Toast.LENGTH_SHORT).show();
    }

}