package com.printer.sdk.demo.ui;

import static android.graphics.pdf.PdfRenderer.Page.RENDER_MODE_FOR_DISPLAY;

import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.graphics.Rect;
import android.graphics.pdf.PdfRenderer;
import android.net.Uri;
import android.os.Bundle;
import android.os.ParcelFileDescriptor;
import android.util.Log;
import android.view.MenuItem;
import android.view.View;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.widget.Toolbar;
import androidx.viewpager2.widget.ViewPager2;

import com.printer.sdk.demo.App;
import com.printer.sdk.demo.R;
import com.printer.sdk.demo.adapter.ViewPagerAdapter;
import com.printer.sdk.demo.bean.Constants;
import com.printer.sdk.demo.bean.SettingParam;
import com.printer.sdk.demo.bean.Triple;
import com.printer.sdk.demo.util.BitmapUtil;
import com.printer.sdk.demo.util.SpUtil;
import com.printer.sdk2.Printer;
import com.printer.sdk2.PrinterConfig;
import com.printer.sdk2.model.Dpi;
import com.printer.sdk2.model.Instruction;
import com.printer.sdk2.model.PaperType;
import com.printer.sdk2.model.PrintMode;
import com.printer.sdk2.model.TearMode;

import java.io.IOException;
import java.util.ArrayList;

import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.functions.Action;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;

public class PdfPreviewActivity extends AppCompatActivity {

    private Toolbar toolbar;
    private ViewPager2 vp;
    private ViewPagerAdapter adapter;
    private TextView tvIndex;
    private Printer printer;
    private SettingParam settingParam;
    private BottomInputDialog inputDialog;


    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_pdf_preview);
        tvIndex = findViewById(R.id.tv_index);

        inputDialog = new BottomInputDialog(this);

        vp = findViewById(R.id.vp);
        adapter = new ViewPagerAdapter(new ArrayList<>());
        vp.setAdapter(adapter);

        toolbar = findViewById(R.id.toolbar);
        toolbar.getMenu().getItem(0).setTitle("设置");
        toolbar.setOnMenuItemClickListener(new Toolbar.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem item) {
                Intent intent = new Intent(PdfPreviewActivity.this, PrinterSettingActivity.class);
                startActivity(intent);
                return false;
            }
        });

        Uri uri = getIntent().getParcelableExtra("pdf");

        try {
            parsePdf(getApplication().getContentResolver().openFileDescriptor(uri, "r"));
        } catch (Exception e) {
            Log.d("PdfPreviewActivity", e.getMessage());
        }

        vp.registerOnPageChangeCallback(new ViewPager2.OnPageChangeCallback() {
            @Override
            public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {
                super.onPageScrolled(position, positionOffset, positionOffsetPixels);
            }

            @Override
            public void onPageSelected(int position) {
                super.onPageSelected(position);
                tvIndex.setText((position + 1) + "/" + adapter.getItemCount());
            }

            @Override
            public void onPageScrollStateChanged(int state) {
                super.onPageScrollStateChanged(state);
            }
        });

        findViewById(R.id.btn_print).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if (printer != null && Printer.getConnectedPrinters().size()>0) {
                    inputDialog.show();
                    inputDialog.setBtnPrintClick(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            if (printer == null || Printer.getConnectedPrinters().size()==0) {
                                Toast.makeText(PdfPreviewActivity.this, "请连接打印机", Toast.LENGTH_SHORT).show();
                                Intent intent = new Intent(PdfPreviewActivity.this, PrinterConnectActivity.class);
                                startActivity(intent);
                                return;
                            }
                            int page = inputDialog.getStartPageIndex();
                            if (page > adapter.getItemCount() || page <= 0) {
                                Toast.makeText(PdfPreviewActivity.this, "不在页面总数范围", Toast.LENGTH_SHORT).show();
                                return;
                            }

                            if (inputDialog.getWidthInput() <= 0) {
                                Toast.makeText(PdfPreviewActivity.this, "宽度输入不合法", Toast.LENGTH_SHORT).show();
                                return;
                            }

                            if (inputDialog.getHeightInput() <= 0) {
                                Toast.makeText(PdfPreviewActivity.this, "高度输入不合法", Toast.LENGTH_SHORT).show();
                                return;
                            }

                            settingParam.setLabelHeight(inputDialog.getHeightInput());
                            settingParam.setLabelWidth(inputDialog.getWidthInput());

                            printPdf(page - 1);
                            inputDialog.dismiss();
                        }
                    });
                } else {
                    Toast.makeText(PdfPreviewActivity.this, "请连接打印机", Toast.LENGTH_SHORT).show();
                    Intent intent = new Intent(PdfPreviewActivity.this, PrinterConnectActivity.class);
                    startActivity(intent);
                }

            }
        });

        if (!Printer.getConnectedPrinters().isEmpty()) {
            printer = Printer.getConnectedPrinters().get(0);
        }

    }

    private void parsePdf(ParcelFileDescriptor fd) {
        if (fd == null) return;
        adapter.getDataList().clear();
        Observable.create(new ObservableOnSubscribe<Triple>() {
                    @Override
                    public void subscribe(ObservableEmitter<Triple> emitter) throws Exception {
                        PdfRenderer mPdfRenderer = new PdfRenderer(fd);
                        int pageCount = mPdfRenderer.getPageCount();

                        for (int i = 0; i < pageCount; i++) {
                            PdfRenderer.Page page = mPdfRenderer.openPage(i);
                            int screenWidth = getApplication().getResources().getDisplayMetrics().widthPixels;
                            Bitmap bitmap = Bitmap.createBitmap(
                                    screenWidth,
                                    (int) (page.getHeight() * screenWidth / (float) page.getWidth()),
                                    Bitmap.Config.ARGB_8888
                            );
                            bitmap.eraseColor(Color.WHITE);
                            page.render(
                                    bitmap,
                                    new Rect(0, 0, bitmap.getWidth(), bitmap.getHeight()),
                                    null,
                                    RENDER_MODE_FOR_DISPLAY
                            );
                            page.close();
                            // emittor.onNext(bitmap)
                            // val pfile = File(tmpFileDir, System.currentTimeMillis().toString() + ".png")
                            //File pfile = File(tmpFileDir, namePrefix + "_" + i + ".png")
                            emitter.onNext(new Triple(i, bitmap));
                        }
                        emitter.onComplete();
                    }
                })
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .doOnNext(new Consumer<Triple>() {
                    @Override
                    public void accept(Triple triple) throws Exception {
                        adapter.getDataList().add(triple);
                    }
                })
                .doOnComplete(new Action() {
                    @Override
                    public void run() throws Exception {
                        adapter.notifyDataSetChanged();
                    }
                })
                .subscribe();
    }

    private void printPdf(int index) {

        int count = adapter.getItemCount();
        Observable.create(new ObservableOnSubscribe<Integer>() {
                    @Override
                    public void subscribe(ObservableEmitter<Integer> emitter) throws Exception {
                        for (int i = index; i < count; i++) {
                            Triple triple = adapter.getDataList().get(i);
                            printBitmap(i, triple.bitmap);
                            Thread.sleep(100);
                        }
                        emitter.onNext(count);
                        emitter.onComplete();
                    }
                })
                .subscribeOn(Schedulers.newThread())
                .observeOn(AndroidSchedulers.mainThread())
                .doOnNext(new Consumer<Integer>() {
                    @Override
                    public void accept(Integer integer) throws Exception {

                    }
                })
                .doOnComplete(new Action() {
                    @Override
                    public void run() throws Exception {

                    }
                })
                .subscribe();


    }

    private SettingParam getSaveSetting() {
        SettingParam settingParam = new SettingParam();
        int dpi = SpUtil.getInt(App.app, Constants.KEY_PRINT_DPI, 0);
        settingParam.setDpi(dpi == 300 ? 300 : 200);

        settingParam.setCompressContent(true);

        //打印速度
        int speed = SpUtil.getInt(App.app, Constants.KEY_PRINT_SPEED, 2);
        settingParam.setPrintSpeed(speed);

        //打印纸张类型
        int paperType = SpUtil.getInt(App.app, Constants.KEY_PRINT_PAPER_TYPE, 1);
        settingParam.setPaperType(paperType);

        //打印浓度
        int concentration = SpUtil.getInt(App.app, Constants.KEY_PRINT_DENSITY, 8);
        settingParam.setPrintConcentration(concentration);

        return settingParam;
    }

    @Override
    protected void onResume() {
        super.onResume();
        settingParam = getSaveSetting();
    }


    private void printBitmap(int index, Bitmap bitmap) {
        PrinterConfig config = new PrinterConfig();
        config.setDpi(settingParam.getDpi() == 300 ? Dpi.DPI_300 : Dpi.DPI_203);
        config.setDensity(settingParam.getPrintConcentration());
        config.setGap(settingParam.getLabelSpace());
        config.setBlackLineHeight(settingParam.getBlineHeight());

        int bitmapWidth = settingParam.getDpi() == SettingParam.DPI_200 ? settingParam.getLabelWidth() * 8 : settingParam.getLabelWidth() * 12;
        float bitmapHeight = (float) bitmap.getHeight() * (float) bitmapWidth / (float)bitmap.getWidth();
        bitmap = BitmapUtil.resizeBitmap(
                bitmap,
                bitmapWidth,
                (int) bitmapHeight
        );
        config.setLabelWidth(settingParam.getDpi() == SettingParam.DPI_200 ? bitmap.getWidth() / 8 : bitmap.getWidth() / 12);
        config.setLabelHeight(settingParam.getDpi() == SettingParam.DPI_200 ? bitmap.getHeight() / 8 : bitmap.getHeight() / 12);
        if (settingParam.getPaperType() == SettingParam.PAPER_TYPE_CONTINUOUS) {
            config.setPaperType(PaperType.PAPER_TYPE_CONTINUOUS);
        } else if (settingParam.getPaperType() == SettingParam.PAPER_TYPE_INTERVAL) {
            config.setPaperType(PaperType.PAPER_TYPE_INTERVAL);
        } else if (settingParam.getPaperType() == SettingParam.PAPER_TYPE_BLACK) {
            config.setPaperType(PaperType.PAPER_TYPE_BLACK_LINE);
        } else {
            config.setPaperType(PaperType.PAPER_TYPE_CONTINUOUS);
        }

        config.setRotation(settingParam.getPrintDirection());
        config.setSpeed(settingParam.getPrintSpeed());
        if (settingParam.getTearMode() == SettingParam.TEAR_TEAR) {
            config.setTearMode(TearMode.TEAR);
        } else if (settingParam.getTearMode() == SettingParam.TEAR_PEEL) {
            config.setTearMode(TearMode.PEEL);
        } else if (settingParam.getTearMode() == SettingParam.TEAR_CUT) {
            config.setTearMode(TearMode.CUT);
        } else if (settingParam.getTearMode() == SettingParam.TEAR_CUT_END) {
            if ((index + 1) == adapter.getDataList().size()) {
                config.setTearMode(TearMode.CUT);
            } else {
                config.setTearMode(TearMode.PEEL);
            }
        }
        config.setxOffset(settingParam.getLevelReference());
        config.setyOffset(settingParam.getVerticalReference());
        config.setPrintCount(1);
        //热转、热敏
        config.setPrintMode(PrintMode.DIRECT_THERMAL);
        config.setSlice(true);
        config.setBeep(settingParam.isSound());
        config.setBitmapShake(false);
        config.setCompressBitmap(settingParam.isCompressContent());
        if (settingParam.getInstruction() == SettingParam.INSTRUCTION_ESC) {
            config.setInstruction(Instruction.ESC);
        } else if (settingParam.getInstruction() == SettingParam.INSTRCUTION_TSC) {
            config.setInstruction(Instruction.TSC);
        } else if (settingParam.getInstruction() == SettingParam.INSTRUCTION_ZPL) {
            config.setInstruction(Instruction.ZPL);
        } else if (settingParam.getInstruction() == SettingParam.INSTRUCTION_CPCL) {
            config.setInstruction(Instruction.CPCL);
        } else {
            config.setInstruction(Instruction.TSC);
        }

        Log.d("PdfPreviewActivity", "index: ${currentLabelIndex}, 指令：${config.instruction}");
        config.setSlice(false);
        if (printer != null && Printer.getConnectedPrinters().size()>0) {
            if (bitmap.getWidth() > 12 && bitmap.getHeight() > 1) {
                try {
                    printer.print(bitmap, 1, config);
                } catch (IOException e) {
                    Log.d("PdfPreviewActivity", "print: " + e.getMessage());
                }
            }
        }
    }
}
